# -*- coding: utf-8 -*-

'''
Created 03-Jan-2025
See VersionInfo in __main__ for current update
'''
import os
from pathlib import Path

from MSPyBentley import *
from MSPyBentleyGeom import *
from MSPyDgnPlatform import *
from MSPyMstnPlatform import *

from la_solutions.configuration_vars import GetReportFolder
from la_solutions.match_with_class import TextSourceIdentity, DescribeTextSource
from la_solutions.point_object_info import PointElementInfo
from la_solutions.version_info import VersionInfo
from la_solutions.dgn_elements import GetElementDescription, EXPOSE_CHILDREN_COUNT

# Two ways of making a smart tuple: NamedTuple is more recent in Python history
from collections import namedtuple
from typing import NamedTuple
# Pandas is a library for handling large data sets and provides CSV and Excel read/write functions
import pandas as pd

# A Python tuple of DGN element handlers    
# Tuples of DGN element handlers to specify search requirements.
# DGN element handlers are documented under MSPyDgnPlatform
# See: https://stackoverflow.com/questions/42385916/inheriting-from-a-namedtuple-base-class

# tuples are a list of handler classes, but they are anonymous
#TextHandlers = (TextElemHandler, TextNodeHandler, TextTableHandler, )
#LineHandlers = (LineHandler, LineStringHandler, )
#ShapeHandlers = (ShapeHandler, )
#ArcHandlers = (ArcHandler, EllipticArcBaseHandler, )
# A circle is an ellipse, having equal axes: there is no separate CircleHandler
#EllipseHandlers = (EllipseHandler, )
# A namedtuple behaves like a tuple and has a name and named fields.
#                          Tuple Name, DGN Element, DPoint3d,  WString
#ntTextInfo =  namedtuple ("TextInfo", "elementId", "text", "origin") 

def GetPointElementInfo (eh: ElementHandle, source: str, sourceId: int, uors: float)->PointElementInfo:
    # Get text or cell information
    # Element ID, Level ID, Text, x, y, z, source, element ID of source
    handler = eh.GetHandler()
    textContent = str()
    
    if isinstance (handler, TextElemHandler):
        text_query = eh.GetITextQuery()
        if text_query.IsTextElement(eh):
            text_part_ids = TextPartIdVector()
            text_query.GetTextPartIds(eh, ITextQueryOptions(), text_part_ids)
            first_text_part = text_part_ids[0]
            text_block = text_query.GetTextPart(eh, first_text_part)
            origin = DPoint3d()           
            origin = text_block.GetUserOrigin()
            textContent += str(text_block.ToString())
            # Convert MicroStation units-of-resolution to master units
            return PointElementInfo (eh.GetElementId(), eh.GetElement ().ehdr.level, textContent,
                origin.x * uors, origin.y * uors, origin.z * uors, source, sourceId)
            
    elif isinstance (handler, TextNodeHandler):
        (source, sourceId) = TextSourceIdentity(eh)
        text_block = handler.GetFirstTextPartValue()
        origin = DPoint3d()           
        origin = text_block.GetUserOrigin()
        textContent += str(text_block.ToString())
        # Convert MicroStation units-of-resolution to master units
        return PointElementInfo (eh.GetElementId(), eh.GetElement ().ehdr.level, textContent, 
            origin.x * uors, origin.y * uors, origin.z * uors, source, sourceId)
       
    return None
     
def GetText (eh: ElementHandle)->str:
    # We've found a text or text-node element and want to extract its text.
    handler = eh.GetHandler()
    content = str()
    if isinstance (handler, TextElemHandler):
        text_query = eh.GetITextQuery()
        if text_query.IsTextElement(eh):
            text_part_ids = TextPartIdVector()
            text_query.GetTextPartIds(eh, ITextQueryOptions(), text_part_ids)
            first_text_part = text_part_ids[0]
            text_block = text_query.GetTextPart(eh, first_text_part)
            word = text_block.ToString()
            content += str(word)
            
    elif isinstance (handler, TextNodeHandler):
        text_block = handler.GetFirstTextPartValue()
        word = text_block.ToString()
        content += str(word)
        
    return content
       
def MakeNamedTupleFromHandlerList (name: str, handlers)->namedtuple:
    # Construct a list of arbitrary length of named classes prior to making a namedtuple  e.g. "c1 c2 c3 ..."
    nHandlers = len(handlers)

    def expand():
        for i in range(0, len(handlers)):
            yield f"c{i}"
            
    args = ""            
    for i in expand():
        args = args + f"{i} "
    
    # Create the tuple factory
    nt_factory = namedtuple (name, args)    
    return nt_factory (*handlers)
    
TextHandlers  = MakeNamedTupleFromHandlerList("words", (TextElemHandler, TextNodeHandler, TextTableHandler, )) 
ComplexElementHandlers = MakeNamedTupleFromHandlerList("Complex", (TextNodeHandler, NormalCellHeaderHandler, )) 

def HarvestPointElements (dgnModel: DgnModel, handlers: tuple)->list:
    """
    Extract text from text elements, text nodes and text nested in cells.
    Returns a list of PointElementInfo.
    """
    modelInfo = dgnModel.GetModelInfo()
    # Convert MicroStation units-of-resolution to master units
    uors: float = 1 / modelInfo.GetUorPerMeter()
    pointInfoList = list()
    for elemRef in dgnModel.GetGraphicElements():
        # Create an element handle and get its Handler
        eh = ElementHandle (elemRef, dgnModel)            
        handler = eh.GetHandler() 
        n = 0
        if isinstance (handler, ComplexElementHandlers):
            (source, sourceId) = TextSourceIdentity(eh)            
            description = GetElementDescription(eh)
            msg = f"[{n}] Found element {description} source={source} ID={sourceId}"
            MessageCenter.ShowDebugMessage(msg, msg, False)
            # Find the text components of a complex element
            component = ChildElemIter(eh, EXPOSE_CHILDREN_COUNT)
            while component.IsValid():
                n += 1
                handler = component.GetHandler()
                description = GetElementDescription(component)
                msg = f"[{n}] Found component {description} source={source} ID={sourceId}"
                MessageCenter.ShowDebugMessage(msg, msg, False)
                if isinstance (handler, TextHandlers):
                    # Element ID, Level ID, Text, x, y, z, source, source ID               
                    pointInfoList.append (GetPointElementInfo (component, source, sourceId, uors))              
                
                component = component.ToNext()
                                
        elif isinstance (handler, TextElemHandler):
            # Element ID, Level ID, Text, x, y, z, source
            (source, sourceId) = TextSourceIdentity(eh)
            pointInfoList.append (GetPointElementInfo (eh, source, 0, uors))
        
    return pointInfoList  
   
def CreateExcelSheetName (preamble, name)->str:
    '''
    Create an Excel sheet name from a preamble and a name, subject to Excel restrictions.
    https://www.google.com/search?client=firefox-b-d&q=excel+valid+sheet+name
    '''
    MAX_LABEL_LENGTH = 31
    labelName = f" '{name}' " # Trailing space after closing quote
    if MAX_LABEL_LENGTH < len(preamble) + len(name):
        #   Reduce preamble
       remainder = MAX_LABEL_LENGTH - len(name) - 2
       return f"{preamble [:remainder]}{labelName}"
    else:
        return preamble + f"{labelName}"
def ExtractPointInfoToExcel(pointInfo: list[PointElementInfo], file_path=''):
    '''
    Write a list of PointElementInfo to an Excel file.
    User can supply a fully-qualified file path, or let this
    function create a file name fron an environment variable and the DGN file name.
    '''
    # Get the active DGN file from the session manager
    dgnFile = ISessionMgr.ActiveDgnFile
    fileName = dgnFile.GetFileName()
    baseName = Path(str(fileName)).stem
    modelName = ISessionMgr.ActiveDgnModelRef.GetDgnModel().GetModelName()
    # Set the file path for the output Excel file
    if file_path:
        excelFile = file_path
    else:
        (valid, path) = GetReportFolder()
        excelFile = os.path.join(os.path.dirname(os.path.abspath(path)), f'{baseName}.xlsx')
    msg = f"ExtractPointInfoToExcel nPoints={len(pointInfo)}"
    MessageCenter.ShowDebugMessage(msg, msg, False)
    msg = f"file '{excelFile}' model '{modelName}'"
    MessageCenter.ShowDebugMessage(msg, msg, False)
    
    # Initialize a Pandas DataFrame to store data
    df = pd.DataFrame(pointInfo, columns=PointElementInfo.ColumnNames())
    # If this is the first instance, create an empty DataFrame with the column names
    if df.empty:
        df = pd.DataFrame(columns=PointElementInfo.ColumnNames())
        msg = f"Assign column names {PointElementInfo.ColumnNames()}"
        MessageCenter.ShowDebugMessage(msg, msg, False)

    #print(df)
    sheetLabel = CreateExcelSheetName("DGN Model", modelName)
    
    # Open the Excel file in write mode using Pandas.
    # This fails is the file is already open in Excel.
    with pd.ExcelWriter(excelFile) as writer:
        # Write the DataFrame to the Excel file
        msg =f'Write to Excel sheet "{sheetLabel}"'
        MessageCenter.ShowDebugMessage(msg, msg, False)
        df.to_excel(writer, index=False, sheet_name=sheetLabel) # trailing space for Excel
        msg = "Write to file OK"
        MessageCenter.ShowDebugMessage(msg, excelFile, False)
        
        # Get a reference to the workbook and worksheet
        workbook  = writer.book
        worksheet = writer.sheets[sheetLabel]
        worksheet.Visible = True
        # Save the workbook (and therefore the entire Excel file)
        workbook.save(excelFile)
        msg = f"Saved DGN point information to file '{excelFile}'"
        MessageCenter.ShowInfoMessage(msg, msg, False)
        
if __name__ == "__main__":  # check if this script is being run directly (not imported as a module)
    vinfo = VersionInfo("Text Exporter", 25, 1, 27, "Export DGN text to Excel")
    MessageCenter.ShowDebugMessage(vinfo.brief, vinfo.verbose, False)
    pointInfoList = list(HarvestPointElements (ISessionMgr.ActiveDgnModelRef.GetDgnModel(), ComplexElementHandlers))
    # With a list of PointElementInfo we can write a CSV or Excel file using 'pandas', another Python module.
    ExtractPointInfoToExcel(pointInfoList)    